/*****************************************************************************
 * prodos/dir.c
 * File and inode operations for directories.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 26-May-2001: Created
 *****************************************************************************/

/*= Kernel Includes =========================================================*/

#include <linux/blkdev.h>

/*= ProDOS Includes =========================================================*/

#include "prodos.h"

/*= Forward Declarations ====================================================*/

/* For prodos_dir_inode_operations. */
struct dentry *prodos_lookup(struct inode *,struct dentry *);
int prodos_create(struct inode *,struct dentry *,int);
int prodos_unlink(struct inode *,struct dentry *);
int prodos_rename(struct inode *,struct dentry *,struct inode *,struct dentry *);

/* For prodos_dir_operations. */
int prodos_readdir(struct file *,void *,filldir_t);

/*= VFS Interface Structures ================================================*/

struct inode_operations prodos_dir_inode_operations = {
	lookup: prodos_lookup,
	create: prodos_create,
	unlink: prodos_unlink,
	rename: prodos_rename
};

struct file_operations prodos_dir_operations = {
	read: generic_read_dir,
	readdir: prodos_readdir
/*	fsync: file_fsync */
};

/*= Private Functions =======================================================*/

/*****************************************************************************
 * prodos_compare_pq()
 * Compares a ProDOS-style string to a qstr. Notice that the return value is
 * different than that of typical string comparison functions. -1 is
 * returned if the strings do not match; PRODOS_DFORK_DATA is returned if the
 * names match exactly; PRODOS_DFORK_META is returned if the name matches the
 * "meta file" name; PRODOS_DFORK_RES is returned if the name matches the
 * resource "fork file" name.
 *****************************************************************************/
inline int prodos_compare_pq(char *s1,struct qstr *s2) {
	SHORTCUT const int len = PRODOS_GET_NLEN(s1);
	int i = 0;
	int qlen = s2->len;
	char fork = 0;

	/* Disregard (but remember) any fork suffix in s2. */
	if (s2->name[qlen - 2] == '_') {
		fork = PRODOS_TOLOWER(s2->name[qlen - 1]);
		qlen -= 2;
	}

	/* Strings can't possibly be equal if lengths are different. */
	if (len != qlen) return -1;

	/* Scan strings for differences. */
	for (i = 0;i < len;i++)
		if (PRODOS_TOUPPER(s1[i + 1]) != PRODOS_TOUPPER(s2->name[i]))
			return -1;

	/* No differences were found; determine type of match. */
	switch (fork) {
		case 0: return PRODOS_DFORK_DATA;
		case 'm': return PRODOS_DFORK_META;
		case 'r': return PRODOS_DFORK_RES;
	}
	return -1;
}

/*****************************************************************************
 * prodos_new_dir_block()
 * Allocate and initialize a new directory block.
 *****************************************************************************/
int prodos_expand_dir(struct super_block *sb,u16 block) {
	int result = 0;
	u16 new_block = 0;
	struct buffer_head *dir_bh = NULL;
	struct prodos_dir_block *dir = NULL;
	
	/* Try to allocate a new block. */
	new_block = prodos_alloc_block(sb);
	if (!new_block) {
		result = -ENOSPC;
		goto out_cleanup;
	}

	/* Read the block. */
	dir_bh = prodos_bread(sb,new_block);
	if (!dir_bh) {
		result = -EIO;
		goto out_cleanup;
	}
	dir = (void*)dir_bh->b_data;

	/* Initialize the block. */
	memset(dir,0,PRODOS_BLOCK_SIZE);
	dir->header.prev = cpu_to_le16(block);
	dir->header.next = 0;
	mark_buffer_dirty(dir_bh);
	result = new_block;

out_cleanup:
	if (dir_bh) prodos_brelse(dir_bh);
	return result;
}

/*****************************************************************************
 * prodos_find_entry()
 * Search directory @inode for an entry. If @name is non-NULL, the directory
 * is searched for an entry with matching name; if @name is NULL, the
 * directory is searched (and possibly expanded) for an unused entry.  Zero is
 * returned on failure; the found entry's inode number is returned on success.
 *****************************************************************************/
unsigned long prodos_find_entry(struct inode *inode,struct qstr *name) {
	unsigned long result = 0;
	u16 prev_block = 0;
	u16 cur_block = PRODOS_INO_DBLK(inode->i_ino);
	u16 cur_entry = 1;
	struct buffer_head *bh = NULL;

	/* Scan the directory for a matching entry. */
	while (cur_block) {
		struct prodos_dir_entry *ent = NULL;

		/* Load next directory block when necessary. */
		if (!bh) {
			/* Load the block. */
			bh = prodos_bread(inode->i_sb,cur_block);
			if (!bh) {
				PRODOS_ERROR_1(sb,"prodos_bread() failed for block %d\n",cur_block);
				goto out_cleanup;
			}

			/* Do a quick integrity check. */
			if (prev_block != le16_to_cpu(((struct prodos_dir_block*)bh->b_data)->header.prev)) {
				PRODOS_ERROR_1(sb,"dir block %i header.prev mismatch",cur_block);
				goto out_cleanup;
			}
		}

		/* Get a pointer to the next entry. */
		ent = &((struct prodos_dir_block*)bh->b_data)->entries[cur_entry];
		if (!name && PRODOS_ISDELETED(*ent)) {
			result = PRODOS_MAKE_INO(cur_block,cur_entry,PRODOS_DFORK_DATA);
			goto out_cleanup;
		}
		else if (name && !PRODOS_ISDELETED(*ent)) {
			int match_type = prodos_compare_pq(ent->name,name);
			switch (match_type) {
				case PRODOS_DFORK_RES:
					if (!PRODOS_ISEXTENDED(*ent)) break;
				case PRODOS_DFORK_DATA:
				case PRODOS_DFORK_META:
					result = PRODOS_ISDIR(*ent) ?
					 PRODOS_MAKE_INO(le16_to_cpu(ent->key_block),PRODOS_DENT_DIR,0):
					 PRODOS_MAKE_INO(cur_block,cur_entry,match_type);
					goto out_cleanup;
			}
		}

		/* Advance to next entry. */
		if (++cur_entry == PRODOS_DIR_ENTS_PER_BLOCK) {
			struct prodos_dir_block *cur_dir = (void*)bh->b_data;

			/* Get next directory block pointer. */
			prev_block = cur_block;
			cur_block = le16_to_cpu(cur_dir->header.next);
			cur_entry = 0;

			/* May be able to extend the directory. */
			if (!name && !cur_block && (PRODOS_INO_DBLK(inode->i_ino) != PRODOS_VOLUME_DIR_BLOCK)) {

				/* XXX: Extend the directory here. */
				/* XXX: Remember to zero the new block's contents. */
				PRODOS_ERROR_0(inode->i_sb,"TODO: directory full; extend");
				goto out_cleanup;
			}

			/* Release previous block. */
			prodos_brelse(bh);
			bh = NULL;
		}
	}

out_cleanup:
	if (bh) {
		prodos_brelse(bh);
		bh = NULL;
	}
	return result;
}

/*****************************************************************************
 * prodos_unlink_all()
 * Unlink (delete) all forks of the ProDOS file represented by an @inode.
 *****************************************************************************/
int prodos_unlink_all(struct inode *inode,struct prodos_dir_entry *ent) {
	int result = 0;
	const u16 dblk = PRODOS_INO_DBLK(inode->i_ino);
	const u8 dent = PRODOS_INO_DENT(inode->i_ino);
	u8 dfrk = PRODOS_INO_DFORK(inode->i_ino);
	struct buffer_head *ext_bh = NULL;

	/* We may have to delete TWO forks. */
	if (PRODOS_ISEXTENDED(*ent)) {
		struct prodos_ext_key_block *ext_block = NULL;

		/* Load the extended directory block. */
		ext_bh = prodos_bread(inode->i_sb,PRODOS_I(inode)->i_key);
		if (!ext_bh) {
			result = -EIO;
			goto out_cleanup;
		}
		ext_block = (void*)ext_bh->b_data;

		/* Free the resource fork. */
		prodos_free_tree_blocks(inode->i_sb,ext_block->res.stype << 4,
		 le16_to_cpu(ext_block->res.key_block),
		 le16_to_cpu(ext_block->res.blocks_used));

		/* Free the data fork. */
		prodos_free_tree_blocks(inode->i_sb,ext_block->data.stype << 4,
		 le16_to_cpu(ext_block->data.key_block),
		 le16_to_cpu(ext_block->data.blocks_used));

		/* Release the extended directory block. */
		prodos_brelse(ext_bh);
		ext_bh = NULL;

		/* Free the extended directory block. */
		prodos_free_block(inode->i_sb,PRODOS_I(inode)->i_key);
	}

	/* ...or we may only have to delete one fork. */
	else {
		prodos_free_tree_blocks(inode->i_sb,PRODOS_I(inode)->i_stype,
		 le16_to_cpu(ent->key_block),
		 le16_to_cpu(ent->blocks_used));
	}

	/* Update this inode. */
	inode->i_nlink--;
	mark_inode_dirty(inode);

	/* Unlink the resource fork inode. */
	if (PRODOS_I(inode)->i_stype == PRODOS_STYPE_EXTENDED) {
		inode = iget(inode->i_sb,PRODOS_MAKE_INO(dblk,dent,PRODOS_DFORK_RES));
		if (inode) {
			down(&inode->i_sem);
			inode->i_nlink--;
			mark_inode_dirty(inode);
			up(&inode->i_sem);
			iput(inode);
		}
	}

	/* We may have been called on either a data fork inode OR a meta file
	 inode.  Unlink the one upon which we were NOT called (the one we were
	 called on has, obviously, already been unlinked.) */
	dfrk = dfrk == PRODOS_DFORK_META ? PRODOS_DFORK_DATA : PRODOS_DFORK_META;
	inode = iget(inode->i_sb,PRODOS_MAKE_INO(dblk,dent,dfrk));
	if (inode) {
		down(&inode->i_sem);
		inode->i_nlink--;
		mark_inode_dirty(inode);
		up(&inode->i_sem);
		iput(inode);
	}

	/* Mark the directory entry as deleted. */
	ent->name[0] &= 0x0f;
	result = 1;

out_cleanup:
	if (ext_bh) prodos_brelse(ext_bh);
	return result;
}

/*****************************************************************************
 * prodos_unlink_res()
 * Unlink (delete) the resource fork of the ProDOS file represented by an
 * @inode.  This is the most involved of the unlink operations.  The resource
 * fork is deleted and its blocks freed, then the file is collapsed into a
 * a regular (non-extended) file by copying the data fork metainformation from
 * the extended directory block into the base directory entry itself.
 *****************************************************************************/
int prodos_unlink_res(struct inode *inode,struct prodos_dir_entry *ent) {
	int result = 0;
	struct buffer_head *key_bh = NULL;
	struct prodos_ext_key_block *key_block = NULL;

	/* Need the key block so we can copy data fork info into the dir entry. */
	key_bh = prodos_bread(inode->i_sb,PRODOS_I(inode)->i_key);
	if (!key_bh) {
		result = -EIO;
		goto out_cleanup;
	}
	key_block = (void*)key_bh->b_data;

	/* Update the directory entry. Note that we don't have to worry about byte
	 order since we're copying from one on-disk entry to another. */
	ent->name[0] &= 0x0f;
	ent->name[0] |= key_block->data.stype << 4;
	ent->key_block = key_block->data.key_block;
	ent->blocks_used = key_block->data.blocks_used;
	memcpy(&ent->eof,&key_block->data.eof,3);

	/* TODO: Update the data and meta inodes here (stype, key block, size, blocks used) */

	/* Free affected disk blocks. */
	prodos_free_tree_blocks(inode->i_sb,key_block->res.stype << 4,
	 le16_to_cpu(key_block->res.key_block),
	 le16_to_cpu(key_block->res.blocks_used));
	prodos_free_block(inode->i_sb,PRODOS_I(inode)->i_key);

	/* The inode is now dirty; also return 1 to cause the directory buffer
	 to be marked dirty. */
	mark_inode_dirty(inode);
	result = 1;

out_cleanup:
	if (key_bh) prodos_brelse(key_bh);
	return result;
}

/*= Interface Functions =====================================================*/

/*****************************************************************************
 * prodos_lookup()
 * Search a directory inode for an entry.
 *****************************************************************************/
struct dentry *prodos_lookup(struct inode *inode,struct dentry *dentry) {
	SHORTCUT struct prodos_sb_info * const psb = PRODOS_SB(inode->i_sb);
	struct dentry *result = NULL;
	unsigned long ino = 0;

	/* Grab directory lock. */
	down(&psb->s_dir_lock);

	/* Attempt to find matching entry and get its inode number. */
	ino = prodos_find_entry(inode,&dentry->d_name);
	if (!ino) {
		result = ERR_PTR(-ENOENT);
		goto out_cleanup;
	}

	/* Get the inode and set up a dentry. */
	inode = iget(inode->i_sb,ino);
	if (!inode) {
		PRODOS_ERROR_1(inode->i_sb,"iget() failed for ino 0x%08x",(int)ino);
		result = ERR_PTR(-EACCES);
		goto out_cleanup;
	}
	dentry->d_op = &prodos_dentry_operations;
	d_add(dentry,inode);

out_cleanup:
	up(&psb->s_dir_lock);
	return result;
}

/*****************************************************************************
 * prodos_create()
 * Create a new entry in directory inode @dir. Note that the VFS has already
 * checked that the filename is unique within the directory.
 *****************************************************************************/
int prodos_create(struct inode *dir,struct dentry *dentry,int mode) {
	SHORTCUT struct prodos_sb_info * psb = PRODOS_SB(dir->i_sb);
	int result = 0;
	unsigned long ino = 0;
	struct buffer_head *dir_bh = NULL;

	PRODOS_ERROR_0(dir->i_sb,"called");

	/* Grab directory lock. */
	down(&psb->s_dir_lock);

	/* Get an unused "inode" in the directory. */
	ino = prodos_find_entry(dir,NULL);
	if (!ino) {
		result = -ENOSPC;
		goto out_cleanup;
	}

	/* Read the directory block. */
/*	dir_bh = prodos_bread(dir->i_sb,PRODOS_INO_DBLK(ino));
	if (!dir_bh) {
		result = -EIO;
		goto out_cleanup;
	}*/

	result = -EPERM;

out_cleanup:
	if (dir_bh) prodos_brelse(dir_bh);
	up(&psb->s_dir_lock);
	return result;
}

/*****************************************************************************
 * prodos_unlink()
 *****************************************************************************/
/* NOTE: The VFS holds the inode->i_zombie semaphore during this call. */
/* The big kernel lock is also held for exactly the duration of this call. */
int prodos_unlink(struct inode *inode,struct dentry *dentry) {
	SHORTCUT struct prodos_sb_info * const psb = PRODOS_SB(inode->i_sb);
	SHORTCUT struct inode * const victim = dentry->d_inode;
	int result = 0;
	struct buffer_head *dir_bh = NULL;
	struct prodos_dir_entry *ent = NULL;

	/* Grab directory lock. */
	down(&psb->s_dir_lock);

	/* Load the directory block. */
	dir_bh = prodos_bread(inode->i_sb,PRODOS_INO_DBLK(victim->i_ino));
	if (!dir_bh) {
		result = -EIO;
		goto out_cleanup;
	}
	ent = &((struct prodos_dir_block*)dir_bh->b_data)->entries[PRODOS_INO_DENT(victim->i_ino)];

	/* Dispatch appropriate unlink function. */
	switch (PRODOS_INO_DFORK(victim->i_ino)) {
	case PRODOS_DFORK_DATA:
		result = prodos_unlink_all(victim,ent);
		break;
	case PRODOS_DFORK_META:
		result = prodos_unlink_all(victim,ent);
		break;
	case PRODOS_DFORK_RES:
		if (!PRODOS_ISEXTENDED(*ent)) {
			result = -ENOENT;
			goto out_cleanup;
		}
		result = prodos_unlink_res(victim,ent);
		break;
	}

	/* The prodos_unlink_*() helper returned 1 if it dirtied the dir block. */
	if (result == 1) {
		mark_buffer_dirty(dir_bh);
		result = 0;
	}

out_cleanup:
	if (dir_bh) prodos_brelse(dir_bh);
	up(&psb->s_dir_lock);
	return result;
}

/*****************************************************************************
 * prodos_rename()
 *****************************************************************************/
int prodos_rename(struct inode *oi,struct dentry *od,struct inode *ni,struct dentry *nd) {
	int result = 0;

	/* Our inode number usage prevents all but in-directory renames. */
	if (oi != ni) {
		result = -EPERM;
		goto out_cleanup;
	}

	result = -EPERM;

out_cleanup:
	return result;
}

/*****************************************************************************
 * prodos_readdir()
 * Standard file operation; reads a directory and returns its entries via
 * the filldir() function supplied by the caller.
 *****************************************************************************/
int prodos_readdir(struct file *filp,void *dirent,filldir_t filldir) {
	SHORTCUT struct super_block * const sb = filp->f_dentry->d_sb;
	SHORTCUT struct prodos_sb_info * const psb = PRODOS_SB(sb);
	SHORTCUT struct inode * const inode = filp->f_dentry->d_inode;
	SHORTCUT struct inode * const parent = filp->f_dentry->d_parent->d_inode;
	SHORTCUT const int show_forks = psb->s_flags & PRODOS_FLAG_SHOW_FORKS;
	int result = 0;
	struct buffer_head *bh = NULL;
	struct prodos_dir_block *dir = NULL;

	/* Handle the special '.' and '..' entries. */
	switch (filp->f_pos) {
		case 0:
			/* Add the '.' entry. */
			if (filldir(dirent,".",1,0,inode->i_ino,DT_DIR) < 0)
				goto out_cleanup;

			/* fpos is simply incremented at this point. */
			filp->f_pos++;
			/* Fall through. */
		case 1:
			/* Add the '..' entry. */
			if (filldir(dirent,"..",2,1,parent->i_ino,DT_DIR) < 0)
				goto out_cleanup;

			/* fpos takes on special format after second entry. */
			filp->f_pos = PRODOS_MAKE_INO(PRODOS_INO_DBLK(inode->i_ino),1,
			 PRODOS_DFORK_DATA);
	}

	/* Grab directory lock. */
	down(&PRODOS_SB(sb)->s_dir_lock);

	/* After the two special entries, filp->f_pos is the "inode number" of the
	 next file (or fork) in the directory. See the PRODOS_INO_*() and
	 PRODOS_MAKE_INO() macros for the format of this value. When all entries
	 have been returned filp->f_pos is set to 3 to signal end-of-directory. */
	while (filp->f_pos != 3) {
		const u16 dblk = PRODOS_INO_DBLK(filp->f_pos);
		const u16 dent = PRODOS_INO_DENT(filp->f_pos);
		const u16 dfrk = PRODOS_INO_DFORK(filp->f_pos);
		struct prodos_dir_entry *ent = NULL;

		/* Load the block if necessary. */
		if (!bh) {
			bh = prodos_bread(sb,dblk);
			if (!bh) {
				result = -EIO;
				goto out_cleanup;
			}
			dir = (struct prodos_dir_block*)bh->b_data;
		}

		/* Get prodos_dir_entry pointer for the next entry. We will skip
		 deleted file entries, and may also skip fork "files" depending upon
		 the PRODOS_FLAG_SHOW_FORKS mount flag. */
		ent = &dir->entries[dent];
		if (!PRODOS_ISDELETED(*ent) && ((dfrk == PRODOS_DFORK_DATA) ||
		 (show_forks && ((dfrk == PRODOS_DFORK_META) ||
		  (PRODOS_ISEXTENDED(*ent) && (dfrk == PRODOS_DFORK_RES)))))) {
			char name[PRODOS_MAX_NAME_LEN + 3] = "";
			int nlen = PRODOS_GET_NLEN(ent->name);
			unsigned int ino = 0;

			/* Copy and modify the name as necessary. */
			memcpy(name,&ent->name,nlen+1);
			prodos_apply_case(name,psb->s_flags & PRODOS_FLAG_LOWERCASE ? 0xffff : ent->case_bits);
			if (dfrk != PRODOS_DFORK_DATA) {
				memcpy(&name[nlen+1],dfrk == PRODOS_DFORK_META ? "_m":"_r",2);
				nlen += 2;
			}

			/* Build inode number. */
			ino = PRODOS_ISDIR(*ent) ?
			 PRODOS_MAKE_INO(le16_to_cpu(ent->key_block),PRODOS_DENT_DIR,PRODOS_DFORK_DATA) :
			 filp->f_pos;

			/* Finally, attempt to return the entry via filldir(). */
			if (filldir(dirent,&name[1],nlen,filp->f_pos,ino,DT_UNKNOWN) < 0)
				goto out_cleanup;
		}

		/* Increment fpos to next entry ino. */
		filp->f_pos = (PRODOS_ISDIR(*ent) || (dfrk == PRODOS_DFORK_RES) ||
		 (!PRODOS_ISEXTENDED(*ent) && dfrk == PRODOS_DFORK_META)) ?
		  PRODOS_MAKE_INO(dblk,dent + 1,PRODOS_DFORK_DATA) :
		  PRODOS_MAKE_INO(dblk,dent,dfrk + 1);
		if (PRODOS_INO_DENT(filp->f_pos) == PRODOS_DIR_ENTS_PER_BLOCK) {
			/* Release current block. */
			prodos_brelse(bh);
			bh = NULL;

			/* Advance f_pos to first entry in the next block. */
			filp->f_pos = PRODOS_MAKE_INO(le16_to_cpu(dir->header.next),0,
			 PRODOS_DFORK_DATA);
			if (!filp->f_pos) filp->f_pos = 3; /* Cause loop to exit. */
		}
	}

/* Release remaining resources and return. */
out_cleanup:
	if (bh) {
		prodos_brelse(bh);
		bh = NULL;
	}
	up(&PRODOS_SB(sb)->s_dir_lock);
	return result;
}

