/*****************************************************************************
 * prodos/prodos.h
 * Includes, definitions, and helpers pertaining to the Linux driver
 * implementation.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 18-May-2001: Created
 *****************************************************************************/
#ifndef __PRODOS_PRODOS_H
#define __PRODOS_PRODOS_H

/*= ProDOS Includes =========================================================*/

#include "prodos_fs.h"

/*= Code Readability Tags ===================================================*/

/* SHORTCUT identifies shortcuts to buried struct members, etc. */
#define SHORTCUT /* */

/*= Preprocessor Constants ==================================================*/

/* Magic number for struct super_block.s_magic. */
#define PRODOS_SUPER_MAGIC				0x065c8160

/* General flags; bits 0-28 of prodos_sb_info.s_flags */
#define PRODOS_FLAG_VERBOSE				0x00000001
#define PRODOS_FLAG_SHOW_FORKS			0x00000002
#define PRODOS_FLAG_LOWERCASE			0x00000004
#define PRODOS_FLAG_CONVERT_CR			0x00000008

/* Inode flags. */
#define PRODOS_I_CRCONV					0x0001

/* 'dfork' values (bits 13-15 of the inode number.) */
#define PRODOS_DFORK_DATA				0
#define PRODOS_DFORK_META				1
#define PRODOS_DFORK_RES				2

/* Special 'dent' value for an entry that refers to a directory itself. */
#define PRODOS_DENT_DIR					0x1fff

/*= Preprocessor Macros =====================================================*/

/* Access to 24-bit little endian integers for ProDOS EOF value. */
#define le24_to_cpup(p)					((u32)(le16_to_cpup((p))|(((char*)(p))[2]<<16)))

/* Simplify access to custom superblock info. */
#define PRODOS_SB(s)					((struct prodos_sb_info *)((s)->u.generic_sbp))
#define PRODOS_I(i)						((struct prodos_inode_info *)&(i)->u.minix_i)

/* Console output. */
#define PRODOS_OUT_0(t,f)				do{printk(t "ProDOS FS:%s:%s: ",__FILE__,__FUNCTION__);printk(f);printk("\n");}while(0)
#define PRODOS_OUT_1(t,f,s1)			do{printk(t "ProDOS FS:%s:%s: ",__FILE__,__FUNCTION__);printk(f,s1);printk("\n");}while(0)
#define PRODOS_OUT_2(t,f,s1,s2)			do{printk(t "ProDOS FS:%s:%s: ",__FILE__,__FUNCTION__);printk(f,s1,s2);printk("\n");}while(0)
#define PRODOS_INFO_0(sb,f)				do{if(PRODOS_SB(sb)->s_flags&PRODOS_FLAG_VERBOSE) PRODOS_OUT_0(KERN_NOTICE,f);}while(0)
#define PRODOS_INFO_1(sb,f,s1)			do{if(PRODOS_SB(sb)->s_flags&PRODOS_FLAG_VERBOSE) PRODOS_OUT_1(KERN_NOTICE,f,s1);}while(0)
#define PRODOS_INFO_2(sb,f,s1,s2)		do{if(PRODOS_SB(sb)->s_flags&PRODOS_FLAG_VERBOSE) PRODOS_OUT_2(KERN_NOTICE,f,s1,s2);}while(0)
#define PRODOS_WARNING_0(sb,f)			PRODOS_OUT_0(KERN_WARNING,f)
#define PRODOS_WARNING_1(sb,f,s1)		PRODOS_OUT_1(KERN_WARNING,f,s1)
#define PRODOS_WARNING_2(sb,f,s1,s2)	PRODOS_OUT_2(KERN_WARNING,f,s1,s2)
#define PRODOS_ERROR_0(sb,f)			PRODOS_OUT_0(KERN_ERR,f)
#define PRODOS_ERROR_1(sb,f,s1)			PRODOS_OUT_1(KERN_ERR,f,s1)
#define PRODOS_ERROR_2(sb,f,s1,s2)		PRODOS_OUT_2(KERN_ERR,f,s1,s2)

/* If a character represents a lowercase letter, convert it to uppercase. */
#define PRODOS_TOUPPER(c)				((c)>='a'&&(c)<='z'?('A'+((c)-'a')):(c))
#define PRODOS_TOLOWER(c)				((c)>='A'&&(c)<='Z'?('a'+((c)-'A')):(c))

/* Get the smallest or largest of two values. */
#define PRODOS_MIN(a,b)					((a)<(b)?(a):(b))
#define PRODOS_MAX(a,b)					((a)>(b)?(a):(b))

/* Access to components of an inode number. */
#define PRODOS_INO_DBLK(ino)			((ino)>>16&0xffff)
#define PRODOS_INO_DFORK(ino)			((ino)>>13&0x0007)
#define PRODOS_INO_DENT(ino)			((ino)&0x1fff)
#define PRODOS_MAKE_INO(b,e,f)			((((b)&0xffff)<<16)|(((f)&0x0007)<<13)|((e)&0x1fff))

/* Determine general type of a directory entry. */
#define PRODOS_ISDELETED(dirent)		(!PRODOS_GET_STYPE((dirent).name))
#define PRODOS_ISDIR(dirent)			(PRODOS_GET_STYPE((dirent).name)==PRODOS_STYPE_DIRECTORY)
#define PRODOS_ISFILE(dirent)			(!PRODOS_ISDELETED(dirent)&&!PRODOS_ISDIR(dirent))
#define PRODOS_ISEXTENDED(dirent)		(PRODOS_GET_STYPE((dirent).name)==PRODOS_STYPE_EXTENDED)

/* Determine the number of data and/or bookkeeping blocks required in order to
 store a file of storage type @st and size @eof. */
#define PRODOS_DATA_BLOCKS_REQ(eof)		(((eof)+(PRODOS_BLOCK_SIZE-1))/PRODOS_BLOCK_SIZE)
#define PRODOS_FS_BLOCKS_REQ(st,eof)	(((st)>0x10?1:0) + ((st)>0x20?(((((eof)+511)>>9)+255)>>8):0))
#define PRODOS_TOTAL_BLOCKS_REQ(st,eof)	(PRODOS_DATA_BLOCKS_REQ(eof)+PRODOS_FS_BLOCKS_REQ(st,eof))

/*= Structure Definitions ===================================================*/

/* super_block information specific to ProDOS filesystems. */
struct prodos_sb_info {
	u32 s_flags;

	/* Items relating to partition location. */
	u32 s_part_start;
	u32 s_part_size;
	u8 s_part[32];

	/* Items relating to the volume bitmap. */
	struct semaphore s_bm_lock;
	struct buffer_head *s_bm_bh[17]; /* one extra to act as NULL terminator */
	u16 s_bm_start;
	int s_bm_free;

	/* Items relating to miscellaneous locking and serialization. */
	struct semaphore s_dir_lock;
};

/* inode information specific to ProDOS "inodes." Note that this structure is
 never actually instantiated. It is simply overlaid upon inode.u.minix_i to
 avoid the need to modify fs.h in the standard Linux source. Obviously, this
 limits the prodos_inode_info structure to a size <= the size of
 struct minix_inode_info. */
struct prodos_inode_info {
	u8 i_flags;
	u8 i_filetype;
	u8 i_stype;
	u16 i_key;
	u16 i_auxtype;
	u8 i_access;
};

/*= Helper Functions ========================================================*/

/* Apply GS/OS character case bits to a filename. */
static inline void prodos_apply_case(char *name,u16 cb) {
	int i = 0;
	int len = PRODOS_GET_NLEN(name);
	if (cb & 0x8000) {
		int b = 0x4000;
		for (i = 0;i < len;i++,b>>=1)
			if (cb & b) name[i+1] = PRODOS_TOLOWER(name[i+1]);
	}
}

/*= Externals ===============================================================*/

/* dir.c */
extern struct inode_operations prodos_dir_inode_operations;
extern struct file_operations prodos_dir_operations;

/* file.c */
extern struct inode_operations prodos_file_inode_operations;
extern struct file_operations prodos_file_operations;
extern struct address_space_operations prodos_address_space_operations;

/* dentry.c */
extern struct dentry_operations prodos_dentry_operations;

/* inode.c */
extern void prodos_read_inode(struct inode *);
extern void prodos_write_inode(struct inode *,int);
extern void prodos_put_inode(struct inode *);

/* super.c */
extern int prodos_count_free_blocks(struct super_block *sb);
extern int prodos_alloc_block(struct super_block *sb);
extern int prodos_free_block(struct super_block *sb,u16 block);
extern int prodos_free_tree_blocks(struct super_block *sb,u8,u16,u16);

/* misc.c */
extern int prodos_check_name(struct qstr *);
extern struct buffer_head *prodos_bread(struct super_block *,int);
extern void prodos_brelse(struct buffer_head *);

#endif
